package iactivegraph;

import java.awt.*;
import java.lang.*;
import java.util.Vector;
import iactivegraph.IGraphFunction;
import iactivegraph.IGraphAverageFunction;
import iactivegraph.IGraphLine;
import iactivegraph.IGraphRangeException;
import iactivegraph.IGraphException;

public class IGraph extends Panel {
	IGraphCanvas	canvas;
	IGraphPanel		controls;
    public IGraph(int fx, int tx, int nx, int scx,
			int fy, int ty, int ny, int scy,
			int fr, int tr, int nr, int scr,
			IGraphFunction f) throws IGraphException {
		if (nx <= 0 || ny <= 0 || nr < 0 || scx <= 0 || scy <= 0 || scr <= 0)
			throw(new IGraphException("Bad constructor value for IGraph"));
		if (tx <= fx || ty <= fy || tr <= fr)
			throw(new IGraphException("Bad range in constructor for IGraph"));
		if (scx != 1 && scx != 10 && scx != 100 && scx != 1000 && scx != 10000)
			throw(new IGraphException("Bad x scale in constructor for IGraph"));
		if (scy != 1 && scy != 10 && scy != 100 && scy != 1000 && scy != 10000)
			throw(new IGraphException("Bad y scale in constructor for IGraph"));
		if (scr != 1 && scr != 10 && scr != 100 && scr != 1000 && scr != 10000)
			throw(new IGraphException("Bad r scale in constructor for IGraph"));
		canvas = new IGraphCanvas(this,
			fx, tx, nx, scx,
			fy, ty, ny, scy,
			fr, tr, nr, scr,
			f
			);
		controls = new IGraphPanel(this);
		setLayout(new BorderLayout());
		add("Center", canvas);
		add("South", controls);
	}

	/**
		Constructor where the result equals the Y parameters.
	*/
    public IGraph(int fx, int tx, int nx, int scx,
			int fy, int ty, int ny, int scy, IGraphAverageFunction f)
				throws Exception {
			this(
				fx, tx, nx, scx,
				fy, ty, ny, scy,
				fy, ty, ny, scy,
				f);
	}
	/**
		Constructor where the result equals the Y parameters.
		The function defaults to the average.
	*/
    public IGraph(int fx, int tx, int nx, int scx,
			int fy, int ty, int ny, int scy) throws Exception {
			this(
				fx, tx, nx, scx,
				fy, ty, ny, scy,
				fy, ty, ny, scy,
				new IGraphAverageFunction(fy, ty, scy));
	}
	public void addLine(int from, int to, int scale, Color color)
			throws IGraphException {
		canvas.addLine(from, to, scale, color);
	}
	public void clear() {
		if (canvas != null) canvas.clear();
	}
	public void setAnnotate(boolean a) {
		controls.annotate.setState(a);
	}
	public String rValToValString(int val) {
		return canvas.rValToValString(val);
	}
	public int	rValToYPos(int val) {
		return canvas.rValToYPos(val);
	}
}
class IGraphPanel extends Panel {
	Button		clear;
	Checkbox	annotate, result;
	IGraph		graph;
	public IGraphPanel(IGraph g) {
		graph = g;
		add(annotate = new Checkbox("Annotate"));
		add(result = new Checkbox("Result"));
		add(clear = new Button("Clean"));
		annotate.setState(true);
		result.setState(true);
	}
    public boolean action(Event ev, Object arg) {
		if (ev.target == clear) {
			graph.clear();
			return true;
		} else if (ev.target == annotate || ev.target == result) {
			graph.canvas.repaint();
			return true;
		}
		return false;
	}
}
class IGraphCanvas extends Canvas {
	IGraph		graph;
	Image		offImage;
	Image		offFrameImage;
	IGraphPoint	selectedPoint;
	Vector		lines;
	Font		scaleFont;
	// Left and bottom borders
	final	int	lb = 20, rb = 20, bb = 10, tb = 20;
	// horizontal and vertical counts
	int			xc, yc, rc;
	int			scalex, scaley, scaler;			// Scale in powers of ten
	int			fromx, fromy, fromr, tox, toy, tor;
	int			xincr, yincr, rincr;
	int			xoffs, yoffs, roffs;
	int			xsize, ysize, rsize;
	int			haxis, vaxis, raxis;
	int			bottom, side;
	IGraphFunction function;

    public IGraphCanvas(IGraph gr, int fx, int tx, int nx, int scx,
			int fy, int ty, int ny, int scy,
			int fr, int tr, int nr, int scr,
			IGraphFunction f) {
		graph = gr;
		xc = nx; yc = ny; rc = nr;
		scalex = scx; scaley = scy; scaler = scr;
		fromx = fx; fromy = fy; fromr = fr; tox = tx; toy = ty; tor = tr;
		xincr = (tox-fromx)/xc; yincr = (toy-fromy)/yc; rincr = (tor-fromr)/rc;
		selectedPoint = null;
		scaleFont = new java.awt.Font("TimesRoman",Font.PLAIN, 10);
		lines = new Vector();
		function = f;
    }
    
	/** Given a result axis value, return the Y position */
	public int	rValToYPos(int val) {
		Dimension	d = size();
		return (bottom)-((val-fromr)*roffs)/rincr;
	}
	/** Given a y axis value, return the Y position */
	public int	valToYPos(int val) {
		Dimension	d = size();
		return (bottom)-((val-fromy)*yoffs)/yincr;
	}
	/** Given a x axis value, return the X position */
	public int	valToXPos(int val) {
		return side+((val-fromx)*xoffs)/xincr;
	}
	/** Given a Y axis position, return the R value */
	public int yPosToRVal(int y) {
		return ((bottom)-y)*rincr/roffs + fromr;
	}
	/** Given a Y axis position, return the value */
	public int yPosToVal(int y) {
		return ((bottom)-y)*yincr/yoffs + fromy;
	}
	/** Given an X axis position, return the value */
	public int xPosToVal(int x) {
		return (x-side)*xincr/xoffs + fromx;
	}
	/** Given a decimal part and a scale, return the string for it (with the .)
	*/
	String decToString(int dec, int scale) {
		if (dec < 0) dec = -dec;
		return (scale>1?
				(("."+
					(
					(scale==10000)?(dec<10?"000":dec<100?"00":dec<1000?"0":"") :
					(scale==1000)?(dec<10?"00":dec<100?"0":"") :
					(scale==100)?(dec<10?"0":"") :
					""
					)
				)+dec):"");
	}
	/** Given a R value, return the value in String form */
	String rValToValString(int val) {
		int	left = val%scaler;
		int dec = left*scaler/scaler;
		int	intpart = val/scaler;
		return (((intpart==0&&val<0)?"-":"")+intpart+decToString(dec, scaler));
	}
	/** Given a Y value, return the value in String form */
	String yValToValString(int val) {
		int	left = val%scaley;
		int dec = left*scaley/scaley;
		int	intpart = val/scaley;
		return (((intpart==0&&val<0)?"-":"")+intpart+decToString(dec, scaley));
	}
	/** Given a Y axis position, return the value in String form */
	String yPosToValString(int y) {
		int	val = yPosToVal(y);
		return yValToValString(val);
	}
	/** Given an X value, return the value in String form */
	String xValToValString(int val) {
		int	left = val%scalex;
		int dec = left*scalex/scalex;
		int	intpart = val/scalex;
		return (((intpart==0&&val<0)?"-":"")+intpart+decToString(dec, scalex));
	}
	/** Given an X axis position, return the value in String form */
	String xPosToValString(int x) {
		int	val = xPosToVal(x);
		return xValToValString(val);
	}
	/**
		Add a line to the graph fromc Y value 'from' to value 'to'.
		The number are assumed to have 'scale' decimals. The
		X points will be printed in the color 'color'
	*/
	public void addLine(int from, int to, int scale, Color color)
			throws IGraphException {
		if (from < fromy || from > toy)
			throw(new IGraphRangeException("addLine from:",fromy,toy,from));
		if (to < fromy || to > toy)
			throw(new IGraphRangeException("addLine to:",fromy,toy,to));
		if (to < from)
			throw(new IGraphException("addLine to<from"));
		IGraphLine	line;
		lines.addElement(line = new IGraphLine(this, color));
		int	yoffs = (to-from)/xc;
		int	x = fromx, y = from;
		for (int i = 0; i <= xc; i++) {
			int ypos = valToYPos(y);
			line.addPoint(valToXPos(x), ypos, x, yPosToVal(ypos));
			x += xincr;
			y += yoffs;
		}
		repaint();
	}
	public void clear() {
		offImage = null;
		lines = null;
		repaint();
	}
	public void paint(Graphics g) {
		update(g);
	}
	public synchronized void update(Graphics g) {
		try {
			Dimension d;
			if (offImage == null) {
				d = size();
				xsize = d.width-lb-rb;
				xoffs = xsize/xc;
				xsize = xoffs*xc;
				ysize = d.height-bb-tb;
				rsize = ysize;
				yoffs = ysize/yc;
				roffs = rsize/rc;
				ysize = yoffs*yc;
				// Bottom and side of the graph
				bottom = ysize+tb;
				side = lb;
				// axis positions
				if (fromx <= 0 && tox > 0) vaxis = valToXPos(0);
				else vaxis = side;
				if (fromy <= 0 && toy > 0) haxis = valToYPos(0);
				else haxis = bottom;


				if (lines == null) lines = new Vector();
				function.initGraph(graph);
//System.out.println("fromy "+fromy+" haxis "+haxis+" toy "+toy);
				offImage = createImage(d.width, d.height);
			}
			if (offFrameImage == null) {
				d = size();
				offFrameImage = createImage(d.width, d.height);
				Graphics frameg = offFrameImage.getGraphics();
				offFramePaint(frameg);
			}
			Graphics offg = offImage.getGraphics();
			offg.drawImage(offFrameImage, 0, 0, this);
			offPaint(offg);
			g.drawImage(offImage, 0, 0, this);
		} catch (java.lang.OutOfMemoryError e) {
			System.out.println("Applet Banner ran out of memory");
			System.exit(1);
		}
	}
	void offFramePaint(Graphics g) {
		Dimension d = size();
		int	i;
		g.clearRect(0, 0, d.width, d.height);

		// Draw the lines
		g.setColor(Color.blue);
		g.drawLine(vaxis, tb, vaxis, bottom);
		g.drawLine(lb, haxis, xsize+lb, haxis);

		g.setFont(scaleFont);
		FontMetrics fm = g.getFontMetrics();

		// Draw the big (full) markers
		int	x = side;
		int	val = fromx;
		for (i = 0; i <= xc; i++) {
			g.drawLine(x, haxis, x, haxis+5);
			String s = xValToValString(val);
			g.drawString(s, x-fm.stringWidth(s)/2, haxis+bb);
			x += xoffs;
			val += xincr;
		}
		int y = bottom;
		val = fromy;
		for (i = 0; i <= yc; i++) {
			g.drawLine(vaxis-5, y, vaxis, y);
			g.drawString(yValToValString(val), vaxis-lb, y-1);
			y -= yoffs;
			val += yincr;
		}
		// Draw the small (half) markers
		x = side+xoffs/2;
		for (i = 0; i < xc; i++) {
			g.drawLine(x, haxis, x, haxis+2);
			x += xoffs;
		}
		y = bottom-yoffs/2;
		for (i = 0; i < yc; i++) {
			g.drawLine(vaxis-2, y, vaxis, y);
			y -= yoffs;
		}

	}
	void offPaint(Graphics g) {
		g.setFont(scaleFont);
		for (int i = 0; i < lines.size(); i++) {
			IGraphLine line = (IGraphLine)lines.elementAt(i);
			line.paint(g, this.graph);
		}
		if (selectedPoint != null) {
			int y = selectedPoint.y;
			g.setColor(Color.black);
			String  s = yPosToValString(y);
			g.drawString(s,
					selectedPoint.x-selectedPoint.rad, y-selectedPoint.rad);
		}
		if (graph.controls.result.getState()) {
			// Draw the result line
			g.setColor(Color.black);
			g.drawLine(side+xsize, tb, side+xsize, bottom);
			int	val = fromr;
			int	y = bottom;
			int	i;
			for (i = 0; i <= rc; i++) {
				g.drawLine(side+xsize+5, y, side+xsize, y);
				g.drawString(rValToValString(val), side+xsize+2, y-1);
				y -= roffs;
				val += rincr;
			}
			y = bottom-roffs/2;
			for (i = 0; i < rc; i++) {
				g.drawLine(side+xsize+2, y, side+xsize, y);
				y -= roffs;
			}
		}
		// This is the meat, print the calculation of all the lines.
		// It can be overriten to generate a different line
		IGraphLine calcline = new IGraphLine(this,
			graph.controls.annotate.getState());
		function.calculate(lines, calcline);
		calcline.paint(g, this.graph);
	}
    public boolean mouseDrag(Event evt, int x, int y) {
		if (selectedPoint != null) {
			// Guard against going of the screen
			if (y < tb) y = tb;
			else if (y > ysize+tb) y = ysize+tb;
			selectedPoint.moveY(y, yPosToVal(y));
			repaint();
		}
		return true;
	}
    public boolean mouseUp(Event evt, int x, int y) {
		if (selectedPoint != null) {
			selectedPoint = null;
			repaint();
			return true;
		}
		return false;
	}
    public boolean mouseExit(Event evt, int x, int y) {
		if (selectedPoint != null) {
			selectedPoint = null;
			repaint();
			return true;
		}
		return false;
	}
    public boolean mouseDown(Event evt, int x, int y) {
		for (int i = lines.size()-1; i >= 0; i--) {
			IGraphLine line = (IGraphLine)lines.elementAt(i);
			selectedPoint = line.found(x, y);
			if (selectedPoint != null) {
				repaint();
				return true;
			}
		}
		return false;
	}
}

